好的，我将完全按照您提供的文档样式和结构，为 `Sensor` 类生成一份详细的 API 接口开发文档。

-----

# Sensor (视频传感器) 接口开发文档

## Sensor 模块概述

`Sensor` 类是对 K230 平台 **VICAP (Video Image Capture) 模块**的 C++ 封装。它提供了一个面向对象的接口，用于配置、控制和捕获来自图像传感器的数据流。

该类管理 VICAP 设备和通道的属性，包括分辨率、像素格式、帧率、镜像、翻转和裁剪等，并提供了启动/停止视频流以及捕获单帧数据的核心功能。

**主要特性：**

  * 自动管理 VICAP 设备 ID 资源。
  * 支持配置通道的分辨率、像素格式、裁剪和缩放。
  * 提供捕获单帧并自动管理帧缓冲区的接口。

-----

## Sensor 类定义

```cpp
class Sensor
{
public:
    // 构造与析构
    Sensor(int csiNum, int reqWidth = 1920, int reqHeight = 1080, int reqFps = 60);
    ~Sensor();

    // 禁用拷贝和移动 (根据提供的代码片段)
    Sensor(const Sensor&)            = delete;
    Sensor& operator=(const Sensor&) = delete;
    Sensor(Sensor&&)                 = delete;
    Sensor& operator=(Sensor&&)      = delete;

    // 核心生命周期
    int reset();
    int run(k_vicap_work_mode mode = VICAP_WORK_ONLINE_MODE);
    int stop();

    // 图像数据捕获
    int snapshot(k_video_frame_info& info, k_vicap_chn channel = VICAP_CHN_ID_0, int timeoutMs = 1000);

    // 属性设置与获取 - 帧大小
    int                  set_framesize(int width, int height, bool crop = false, k_vicap_chn channel = VICAP_CHN_ID_0);
    int                  set_framesize(int width, int height, int crop_x, int crop_y, int crop_width, int crop_height, k_vicap_chn channel = VICAP_CHN_ID_0);
    std::tuple<int, int> framesize(k_vicap_chn channel = VICAP_CHN_ID_0) const;

    // 属性设置与获取 - 像素格式
    int                 set_pixformat(SensorPixelFormat pixelFormat, k_vicap_chn channel = VICAP_CHN_ID_0);
    SensorPixelFormat   pixformat(k_vicap_chn channel = VICAP_CHN_ID_0) const;

    // 属性设置与获取 - 镜像与翻转
    int  set_hmirror(bool enable = false);
    bool hmirror();
    int  set_vflip(bool enable = false);
    bool vflip();

    // 静态工具函数
    static k_pixel_format to_k_pixel_format(SensorPixelFormat pixel_format);

    int _dev_id();
    int _set_alignment(int alignment, k_vicap_chn channel = VICAP_CHN_ID_0);
    int _set_fps(int fps, k_vicap_chn channel = VICAP_CHN_ID_0);
};
```

-----

## Sensor 类公共 API 接口说明

### 1\. 枚举类型 (Constants)

#### `SensorPixelFormat`

| Enum Value | 对应的 K230 MPI 格式 (`k_pixel_format`) | 描述 |
| :--- | :--- | :--- |
| `InvaildPixelFormat` | `PIXEL_FORMAT_BUTT` | 无效/默认格式。|
| `GRAYSCALE` | `PIXEL_FORMAT_YUV_SEMIPLANAR_420` | 灰度格式（映射到 YUV420SP）。|
| `RGB565` | `PIXEL_FORMAT_RGB_888` | RGB565 格式（映射到 packed RGB888）。|
| `RGB888` | `PIXEL_FORMAT_RGB_888` | RGB888 格式（packed）。|
| `RGB888P` | `PIXEL_FORMAT_RGB_888_PLANAR` | RGB888 分量式（Planar）格式。|
| `YUV420SP` | `PIXEL_FORMAT_YUV_SEMIPLANAR_420` | YUV 420 半平面格式。|

-----

### 2\. 构造与析构

#### `Sensor` 类构造函数

```cpp
Sensor(int csiNum, int reqWidth = 1920, int reqHeight = 1080, int reqFps = 60);
```

**功能：**
初始化 `Sensor` 对象，分配一个 VICAP 设备 ID，并存储初始的探针配置。

**参数：**

| 参数名 | 类型 | 描述 |
| :--- | :--- | :--- |
| `csiNum` | `int` | 要使用的 CSI 接口编号。|
| `reqWidth` | `int` | 期望的初始**探针宽度**。默认值 `1920`。|
| `reqHeight` | `int` | 期望的初始**探针高度**。默认值 `1080`。|
| `reqFps` | `int` | 期望的初始**探针帧率**。默认值 `60`。|

**异常：**

  * 如果 VICAP 设备 ID 超过最大限制，可能抛出 `std::runtime_error`。

#### `Sensor` 类析构函数

```cpp
~Sensor();
```

**功能：**
销毁 `Sensor` 对象，释放其占用的 VICAP 设备 ID 资源。

-----

### 3\. 核心生命周期 API

| API 方法 | 签名 | 功能描述 |
| :--- | :--- | :--- |
| **`reset`** | `int reset();` | **功能：** 根据构造函数中的配置，探测并初始化传感器。此函数必须在设置通道属性和调用 `run()` 之前执行。<br>**返回值：** `0` 成功，`-1` 失败（如传感器探测失败）。|
| **`run`** | `int run(k_vicap_work_mode mode = VICAP_WORK_ONLINE_MODE);` | **功能：** 配置 VICAP 设备和所有已启用的通道属性，并启动 VICAP 视频流。<br>**参数：** `mode` 指定 VICAP 工作模式，默认为在线模式。<br>**返回值：** `0` 成功，`-1` 失败（如未调用 `reset()` 或启动流失败）。|
| **`stop`** | `int stop();` | **功能：** 停止 VICAP 视频流，并对 VICAP 设备进行去初始化操作。<br>**返回值：** `0` 成功，`-1` 失败（如未处于运行状态）。|

-----

### 4\. 图像数据操作 API

#### 获取指定输出张量数据

```cpp
int snapshot(k_video_frame_info& info, k_vicap_chn channel = VICAP_CHN_ID_0, int timeoutMs = 1000);
```

**功能：**
从指定的 VICAP **通道**捕获（dump）一帧视频数据。如果该通道上一次捕获的帧未释放，会自动先释放旧帧。

**参数：**

| 参数名 | 类型 | 描述 |
| :--- | :--- | :--- |
| `info` | `k_video_frame_info&` | 输出参数，成功时将填充捕获到的帧信息（包括缓冲区地址和大小）。|
| `channel` | `k_vicap_chn` | 捕获数据的通道索引。默认值 `VICAP_CHN_ID_0`。|
| `timeoutMs` | `int` | 捕获等待的超时时间（毫秒）。默认值 `1000`。|

**返回值：**

  * `0`：捕获成功。
  * `-1`：捕获失败（例如，未调用 `run()` 或超时）。

-----

### 5\. 属性配置与查询 API

#### 帧大小和裁剪

| API 方法 | 签名 | 功能描述 |
| :--- | :--- | :--- |
| **`set_framesize` (Auto Crop)** | `int set_framesize(int width, int height, bool crop = false, k_vicap_chn channel = VICAP_CHN_ID_0);` | **功能：** 设置通道的输出帧大小。<br>如果 `crop` 为 `true`，系统将根据目标宽高比自动计算一个居中裁剪窗口。<br>如果 `crop` 为 `false`，则尝试使用最大的输入窗口。<br>**返回值：** `0` 成功，`-1` 失败。|
| **`set_framesize` (Manual Crop)** | `int set_framesize(int width, int height, int crop_x, int crop_y, int crop_width, int crop_height, k_vicap_chn channel = VICAP_CHN_ID_0);` | **功能：** 设置通道的输出帧大小，并**手动指定**在 sensor 输入上进行裁剪的区域 (`crop_x`等参数)。<br>**返回值：** `0` 成功，`-1` 失败。|
| **`framesize`** | `std::tuple<int, int> framesize(k_vicap_chn channel = VICAP_CHN_ID_0) const;` | **功能：** 获取指定通道当前配置的输出帧大小 (`width`, `height`)。<br>**返回值：** `std::tuple<width, height>`。|

#### 像素格式

| API 方法 | 签名 | 功能描述 |
| :--- | :--- | :--- |
| **`set_pixformat`** | `int set_pixformat(SensorPixelFormat pixelFormat, k_vicap_chn channel = VICAP_CHN_ID_0);` | **功能：** 使用 `SensorPixelFormat` 枚举设置指定通道的输出像素格式。<br>**返回值：** `0` 成功，`-1` 失败。|
| **`pixformat`** | `SensorPixelFormat pixformat(k_vicap_chn channel = VICAP_CHN_ID_0) const;` | **功能：** 获取指定通道当前配置的像素格式。<br>**返回值：** `SensorPixelFormat` 枚举值。|

#### 镜像与翻转 (Sensor 级别)

| API 方法 | 签名 | 功能描述 |
| :--- | :--- | :--- |
| **`set_hmirror`** | `int set_hmirror(bool enable = false);` | **功能：** 设置是否启用**水平镜像**（左右翻转）。此配置会在 `reset()` 时生效。<br>**返回值：** `0` 成功，`-1` 失败。|
| **`hmirror`** | `bool hmirror();` | **功能：** 获取当前水平镜像的配置状态。<br>**返回值：** `bool`，`true` 为启用。|
| **`set_vflip`** | `int set_vflip(bool enable = false);` | **功能：** 设置是否启用**垂直翻转**（上下翻转）。此配置会在 `reset()` 时生效。<br>**返回值：** `0` 成功，`-1` 失败。|
| **`vflip`** | `bool vflip();` | **功能：** 获取当前垂直翻转的配置状态。<br>**返回值：** `bool`，`true` 为启用。|

#### 静态工具函数

| API 方法 | 签名 | 功能描述 |
| :--- | :--- | :--- |
| **`to_k_pixel_format`** | `static k_pixel_format to_k_pixel_format(SensorPixelFormat pixel_format);` | **功能：** 将自定义的 `SensorPixelFormat` 枚举值转换为 K230 MPI 库使用的原生 `k_pixel_format` 类型。<br>**返回值：** `k_pixel_format`。|
