/*
  Copyright (c) 2016 Arduino LLC.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#pragma once

#include <inttypes.h>
#include "Stream.h"
#include <pthread.h>
#include <functional>
#include "api/HardwareI2C.h"

#include "drv_i2c.h"

#ifndef I2C_BUFFER_LENGTH
#define I2C_BUFFER_LENGTH 128  // Default size, if none is set using Wire::setBuffersize(size_t)
#endif

namespace arduino {

class TwoWire : public HardwareI2C {
protected:
    drv_i2c_inst_t* i2c;
    uint8_t num;
    int8_t sda;
    int8_t scl;

    size_t bufferSize;
    uint8_t *rxBuffer;
    size_t rxIndex;
    size_t rxLength;

    uint8_t *txBuffer;
    size_t txLength;
    uint16_t txAddress;

    bool nonStop;
#if !CONFIG_DISABLE_HAL_LOCKS
    pthread_mutex_t _lock;
    pthread_mutex_t *lock;
    pthread_t curr_tid;
#endif
private:
#if SOC_I2C_SUPPORT_SLAVE
    bool is_slave;
    std::function<void()> user_onRequest;
    std::function<void(int)> user_onReceive;
    static void onRequestService(uint8_t, void *);
    static void onReceiveService(uint8_t, uint8_t *, size_t, bool, void *);
#endif
    bool initPins(int sdaPin, int sclPin);
    bool allocateWireBuffer();
    void freeWireBuffer();

public:
    TwoWire(uint8_t bus_num);
    ~TwoWire();

    bool begin() override final {
        return begin(-1, -1);
    }

    bool begin(uint8_t address) override final {
#if SOC_I2C_SUPPORT_SLAVE
        return begin(address, -1, -1, 0);
#else
        printf("I2C slave is not supported temporarily\n");
        return false;
#endif
    }

    bool end() override;

    uint8_t getBusNum();

    bool setClock(uint32_t freq) override;

    void beginTransmission(uint8_t address) override;
    uint8_t endTransmission(bool stopBit) override;
    uint8_t endTransmission() override;

    size_t requestFrom(uint8_t address, size_t len, bool stopBit) override;
    size_t requestFrom(uint8_t address, size_t len) override;

    void onReceive(const std::function<void(int)> &) override;
    void onRequest(const std::function<void()> &) override;

    //call setPins() first, so that begin() can be called without arguments from libraries
    bool setPins(int sda, int scl);

    bool begin(int sda, int scl, uint32_t frequency = 100000);  // returns true, if successful init of i2c bus
#if SOC_I2C_SUPPORT_SLAVE
    bool begin(uint8_t slaveAddr, int sda, int scl, uint32_t frequency);
#endif /* SOC_I2C_SUPPORT_SLAVE */

    size_t setBufferSize(size_t bSize);

    void setTimeOut(uint16_t timeOutMillis);  // default timeout of i2c transactions is 50ms
    uint16_t getTimeOut();

    uint32_t getClock();

    size_t write(uint8_t) override;
    size_t write(const uint8_t *, size_t) override;
    int available() override;
    int read() override;
    int peek() override;
    void flush() override;

#if SOC_I2C_SUPPORT_SLAVE
    size_t slaveWrite(const uint8_t *, size_t);
#endif /* SOC_I2C_SUPPORT_SLAVE */
};
extern TwoWire Wire;
extern TwoWire Wire1;
extern TwoWire Wire2;
extern TwoWire Wire3;
extern TwoWire Wire4;
}
