## Arduino 模拟 I/O API 手册

### 1. 概述

模拟 I/O 功能允许读取模拟信号和输出 PWM 信号。K230 内置 ADC 支持多通道模拟输入，分辨率可配置为 8-12 位。PWM 输出通过硬件 PWM 模块实现，支持 6 个独立通道，可用于控制 LED 亮度、电机速度等应用。

### 2. API 介绍

#### 2.1 `analogRead` 函数

```cpp
int analogRead(int channel)
```

读取指定 ADC 通道的模拟值。

**参数**

- `channel`: ADC 通道号

**返回值**
ADC 转换结果，范围取决于当前分辨率设置（默认 10 位：0-1023）

**注意**

- 函数内部会自动初始化和去初始化 ADC
- 读取失败返回 -1

#### 2.2 `analogReadResolution` 函数

```cpp
void analogReadResolution(int resolution)
```

设置 ADC 读取分辨率。

**参数**

- `resolution`: 分辨率位数（8-12）

**返回值**
无

**注意**

- 默认分辨率为 10 位

#### 2.3 `analogWrite` 函数

```cpp
void analogWrite(pin_size_t pinNumber, int value)
```

在指定引脚输出 PWM 信号。

**参数**

- `pinNumber`: 引脚编号（必须支持 PWM 功能）
- `value`: PWM 占空比值，范围取决于分辨率设置（默认 8 位：0-255）

**返回值**
无

**注意**

- 不是所有引脚都支持 PWM 输出
- 函数会自动查找并配置合适的 PWM 通道
- PWM 频率默认为 1000Hz

#### 2.4 `analogWriteResolution` 函数

```cpp
void analogWriteResolution(uint8_t resolution)
```

设置 PWM 输出分辨率。

**参数**

- `resolution`: 分辨率位数

**返回值**
无

**注意**
默认分辨率为 8 位（0-255）

#### 2.5 `analogWriteFrequency` 函数

```cpp
void analogWriteFrequency(pin_size_t pinNumber, uint32_t freq)
```

设置指定引脚的 PWM 频率。

**参数**

- `pinNumber`: 引脚编号
- `freq`: PWM 频率（Hz）

**返回值**
无

**注意**

- 频率不能为 0
- 设置会影响后续的 `analogWrite()` 调用

### 3. 使用说明

1. **ADC 特性**：
   - 支持 8-12 位分辨率
   - 自动管理 ADC 初始化和释放
   - 硬件 ADC 分辨率为 12 位

2. **PWM 特性**：
   - 6 个硬件 PWM 通道
   - 自动分配 PWM 通道到引脚
   - 支持频率和占空比独立控制

3. **引脚映射**：
   - 不是所有引脚都支持 PWM
   - 函数会自动检测引脚的 PWM 能力

4. **性能考虑**：
   - `analogRead()` 每次调用都会初始化/去初始化 ADC
   - PWM 设置后持续输出，无需重复调用

### 4. 示例程序

```cpp
#include "Arduino.h"

const int ledPin = 52;     // LED连接到GPIO52（支持PWM4）

void setup() {
    Serial.begin(115200);
    delay(1000);
    
    Serial.println("PWM LED Control Test");
    Serial.println("GPIO52 -> PWM4");
    Serial.println("==================");
}

void loop() {
    Serial.println("Testing different PWM frequencies (50% duty):");
    int frequencies[] = {1, 5, 10, 25};
    
    for (int i = 0; i < sizeof(frequencies)/sizeof(frequencies[0]); i++) {
        Serial.print("   Frequency: ");
        Serial.print(frequencies[i]);
        Serial.println(" Hz");
        
        analogWriteFrequency(ledPin, frequencies[i]);
        analogWrite(ledPin, 128);  // 50%占空比
        delay(3000);
    }
    
    Serial.println("\nTest cycle completed. Restarting...\n");
    delay(2000);
}
```
