## Arduino Timer 模块 API 手册

### 1. 概述

K230 内部集成了六个硬件定时器模块，支持单次触发和周期性定时模式。Arduino环境下提供了标准的定时器接口，支持中断回调功能和灵活的配置选项。

### 2. API 介绍

HardwareTimer 类位于 `arduino` 命名空间中，提供了六个预定义的定时器实例：`Timer0` 至 `Timer5`。

#### 2.1 构造函数

定时器实例已经预定义，无需手动构造：

- `Timer0` - 定时器0实例
- `Timer1` - 定时器1实例
- `Timer2` - 定时器2实例
- `Timer3` - 定时器3实例
- `Timer4` - 定时器4实例
- `Timer5` - 定时器5实例

#### 2.2 `begin` 方法

```cpp
bool begin(uint32_t freq_hz = 12500000)
```

初始化定时器模块。

**参数**

- `freq_hz`: 定时器频率（Hz），默认12.5MHz，范围[12500000 ~ 100000000]

**返回值**

- `true`: 初始化成功
- `false`: 初始化失败

#### 2.3 `end` 方法

```cpp
bool end()
```

关闭定时器模块并释放资源。

**参数**
无

**返回值**

- `true`: 关闭成功
- `false`: 关闭失败

#### 2.4 `start` 方法

```cpp
bool start()
```

启动定时器。

**参数**
无

**返回值**

- `true`: 启动成功
- `false`: 启动失败

#### 2.5 `stop` 方法

```cpp
bool stop()
```

停止定时器。

**参数**
无

**返回值**

- `true`: 停止成功
- `false`: 停止失败

#### 2.6 `restart` 方法

```cpp
bool restart()
```

重新启动定时器。

**参数**
无

**返回值**

- `true`: 重启成功
- `false`: 重启失败

#### 2.7 `setMode` 方法

```cpp
bool setMode(bool auto_reload)
```

设置定时器工作模式。

**参数**

- `auto_reload`: true为周期模式，false为单次触发模式

**返回值**

- `true`: 设置成功
- `false`: 设置失败

#### 2.8 `setFrequency` 方法

```cpp
bool setFrequency(uint32_t freq_hz)
```

设置定时器频率。定时器必须处于停止状态才能设置。

**参数**

- `freq_hz`: 定时器频率（Hz），范围[12500000 ~ 100000000]

**返回值**

- `true`: 设置成功
- `false`: 设置失败

#### 2.9 `setPeriod` 方法

```cpp
bool setPeriod(uint32_t period_ms)
```

设置定时器周期。定时器必须处于停止状态才能设置。

**参数**

- `period_ms`: 定时周期（毫秒）

**返回值**

- `true`: 设置成功
- `false`: 设置失败

#### 2.10 `setAlarm` 方法

```cpp
bool setAlarm(uint32_t alarm_ms)
```

设置定时器报警时间（等同于setPeriod）。

**参数**

- `alarm_ms`: 报警时间（毫秒）

**返回值**

- `true`: 设置成功
- `false`: 设置失败

#### 2.11 `getFrequency` 方法

```cpp
uint32_t getFrequency()
```

获取当前定时器频率。

**参数**
无

**返回值**
定时器频率（Hz），失败返回0

#### 2.12 `getPeriod` 方法

```cpp
uint32_t getPeriod()
```

获取当前定时器周期。

**参数**
无

**返回值**
定时器周期（毫秒）

#### 2.13 `getTimerId` 方法

```cpp
uint8_t getTimerId()
```

获取定时器ID。

**参数**
无

**返回值**
定时器ID（0-5）

#### 2.14 `isStarted` 方法

```cpp
bool isStarted()
```

获取定时器运行状态。

**参数**
无

**返回值**

- `true`: 定时器正在运行
- `false`: 定时器已停止

#### 2.15 `attachInterrupt` 方法

```cpp
void attachInterrupt(const std::function<void()>& callback)
```

绑定定时器中断回调函数。定时器必须处于停止状态才能绑定。

**参数**

- `callback`: 中断回调函数

**返回值**
无

#### 2.16 `attachInterruptArg` 方法

```cpp
void attachInterruptArg(const std::function<void(void*)>& callback, void* arg)
```

绑定带参数的定时器中断回调函数。定时器必须处于停止状态才能绑定。

**参数**

- `callback`: 中断回调函数
- `arg`: 传递给回调函数的参数

**返回值**
无

#### 2.17 `detachInterrupt` 方法

```cpp
void detachInterrupt()
```

解除定时器中断回调函数。定时器必须处于停止状态才能解除。

**参数**
无

**返回值**
无

---

### 3. 示例程序

```cpp
#include "Arduino.h"

using namespace arduino;

#define LED_PIN 52

// 测试状态
enum TestPhase {
    TEST_BASIC,
    TEST_MULTI,
    TEST_DONE
};

TestPhase currentPhase = TEST_BASIC;
uint32_t phaseStartTime = 0;

// 计数器
volatile uint32_t timer0_count = 0;
volatile uint32_t timer1_count = 0;
volatile uint32_t timer2_count = 0;
volatile uint32_t timer3_count = 0;
volatile bool led_state = false;

// 基础测试回调
void basicCallback() {
    timer0_count++;
    led_state = !led_state;
    digitalWrite(LED_PIN, led_state);
}

// 多定时器测试回调
void timer0Callback() {
    timer0_count++;
}

void timer1Callback() {
    timer1_count++;
}

void timer2Callback() {
    timer2_count++;
}

void timer3Callback() {
    timer3_count++;
    led_state = !led_state;
    digitalWrite(LED_PIN, led_state);
}

void setup() {
    Serial.begin(115200);
    delay(1000);

    Serial.println("\n=== K230 Timer Test (Quick Version) ===");
    Serial.println("Tests: Basic functionality & Multi-timer");
    Serial.println("Total test time: ~15 seconds\n");

    pinMode(LED_PIN, OUTPUT);
    digitalWrite(LED_PIN, LOW);

    // 初始化Timer0
    if (!Timer0.begin()) {
        Serial.println("ERROR: Timer0 init failed!");
        while(1);
    }

    runBasicTest();
}

void loop() {
    uint32_t elapsed = millis() - phaseStartTime;

    switch(currentPhase) {
    case TEST_BASIC:
        if (elapsed >= 5000) {  // 5秒基础测试
            Serial.print("Basic test complete. Triggers: ");
            Serial.println(timer0_count);
            currentPhase = TEST_MULTI;
            runMultiTimerTest();
        }
        break;

    case TEST_MULTI:
        if (elapsed >= 10000) {  // 10秒多定时器测试
            showResults();
            currentPhase = TEST_DONE;
        } else if (elapsed % 2000 == 0) {  // 每2秒显示一次进度
            Serial.print(".");
        }
        break;

    case TEST_DONE:
        // 测试完成，空循环
        break;
    }

    delay(100);
}

void runBasicTest() {
    Serial.println("1. Basic Timer Test");
    Serial.println("   - Periodic mode: 200ms");
    Serial.println("   - One-shot test: 2s delay");

    // 测试周期模式
    Timer0.setMode(true);
    Timer0.setPeriod(200);
    Timer0.attachInterrupt(basicCallback);
    Timer0.start();

    Serial.println("   LED blinking at 5Hz...");
    delay(3000);  // 运行3秒

    // 测试单次模式
    Timer0.stop();
    timer0_count = 0;
    Timer0.setMode(false);
    Timer0.setAlarm(2000);
    Serial.println("Test one shot timer");
    Timer0.attachInterrupt([]() {
                           Serial.println("   One-shot triggered!");
                           timer0_count++;
                           });
    Timer0.start();

    phaseStartTime = millis();
}

void runMultiTimerTest() {
    Serial.println("\n2. Multi-Timer Test");
    Serial.println("   Timer0: 100ms  (10Hz)");
    Serial.println("   Timer1: 250ms  (4Hz)");
    Serial.println("   Timer2: 500ms  (2Hz)");
    Serial.println("   Timer3: 1000ms (1Hz)");
    Serial.print("   Running");

    // 重置所有计数器
    timer0_count = 0;
    timer1_count = 0;
    timer2_count = 0;
    timer3_count = 0;

    // 停止Timer0
    Timer0.stop();

    // 初始化其他定时器
    if (!Timer1.begin()) Serial.println("\n   WARNING: Timer1 init failed");
    if (!Timer2.begin()) Serial.println("\n   WARNING: Timer2 init failed");
    if (!Timer3.begin()) Serial.println("\n   WARNING: Timer3 init failed");

    // 配置所有定时器
    Timer0.setMode(true);
    Timer0.setPeriod(100);
    Timer0.attachInterrupt(timer0Callback);

    Timer1.setMode(true);
    Timer1.setPeriod(250);
    Timer1.attachInterrupt(timer1Callback);

    Timer2.setMode(true);
    Timer2.setPeriod(500);
    Timer2.attachInterrupt(timer2Callback);

    Timer3.setMode(true);
    Timer3.setPeriod(1000);
    Timer3.attachInterrupt(timer3Callback);

    // 启动所有定时器
    phaseStartTime = millis();
    Timer0.start();
    Timer1.start();
    Timer2.start();
    Timer3.start();
}

void showResults() {
    Serial.println("\n\n=== Test Results ===");

    // 停止所有定时器
    Timer0.stop();
    Timer1.stop();
    Timer2.stop();
    Timer3.stop();

    // 显示结果
    Serial.println("Expected vs Actual triggers (10s test):");

    Serial.print("Timer0 (100ms): ");
    Serial.print("Expected ~100, Actual ");
    Serial.print(timer0_count);
    Serial.println(abs((int)timer0_count - 100) <= 2 ? " ✓" : " ✗");

    Serial.print("Timer1 (250ms): ");
    Serial.print("Expected ~40,  Actual ");
    Serial.print(timer1_count);
    Serial.println(abs((int)timer1_count - 40) <= 2 ? " ✓" : " ✗");

    Serial.print("Timer2 (500ms): ");
    Serial.print("Expected ~20,  Actual ");
    Serial.print(timer2_count);
    Serial.println(abs((int)timer2_count - 20) <= 2 ? " ✓" : " ✗");

    Serial.print("Timer3 (1000ms): ");
    Serial.print("Expected ~10,  Actual ");
    Serial.print(timer3_count);
    Serial.println(abs((int)timer3_count - 10) <= 1 ? " ✓" : " ✗");

    // 计算精度
    float accuracy0 = (timer0_count / 100.0) * 100;
    float accuracy1 = (timer1_count / 40.0) * 100;
    float accuracy2 = (timer2_count / 20.0) * 100;
    float accuracy3 = (timer3_count / 10.0) * 100;

    Serial.println("\nTimer Accuracy:");
    Serial.print("Timer0: "); Serial.print(accuracy0, 1); Serial.println("%");
    Serial.print("Timer1: "); Serial.print(accuracy1, 1); Serial.println("%");
    Serial.print("Timer2: "); Serial.print(accuracy2, 1); Serial.println("%");
    Serial.print("Timer3: "); Serial.print(accuracy3, 1); Serial.println("%");

    float avgAccuracy = (accuracy0 + accuracy1 + accuracy2 + accuracy3) / 4;
    Serial.print("\nAverage accuracy: ");
    Serial.print(avgAccuracy, 1);
    Serial.println("%");

    Serial.println("\n=== All tests completed! ===");

    // 关闭定时器
    Timer1.end();
    Timer2.end();
    Timer3.end();

    // 关闭LED
    digitalWrite(LED_PIN, LOW);
}
```
