# AI2D 模块接口文档

## AI2D 概述

`AI2D` 类是对 `nncase` 的 AI2D 图像预处理模块的封装，提供了裁剪、缩放、填充、仿射变换、数据类型设置、移位等功能，便于在推理前对图像进行标准化处理。

**主要特性：**

* 支持设置输入输出数据类型和格式
* 提供裁剪、缩放、填充、仿射、移位操作
* 构建 AI2D 流程并执行
* 封装异常处理，保证接口调用安全
* 禁止拷贝构造与赋值，支持移动构造与移动赋值

---

## AI2D类定义

```cpp
class AI2D
{
public:
    explicit AI2D();
    ~AI2D();

    AI2D(const AI2D&) = default;
    AI2D& operator=(const AI2D&) = default;

    AI2D(AI2D&&) noexcept = default;
    AI2D& operator=(AI2D&&) noexcept = default;

    void set_ai2d_dtype(ai2d_format input_format,
                        ai2d_format output_format,
                        typecode_t input_dtype,
                        typecode_t output_dtype);

    void set_crop(size_t x, size_t y, size_t width, size_t height);

    void set_resize(ai2d_interp_method interp_method,
                    ai2d_interp_mode interp_mode);

    void set_pad(const vector<int>& pad,
                 ai2d_pad_mode pad_mode,
                 const vector<int>& pad_value);

    void set_affine(ai2d_interp_method interp_method,
                    int cord_round,
                    int bound_ind,
                    int bound_val,
                    int bound_smooth,
                    const vector<float>& affine_matrix);

    void set_shift(int shift_val);

    void build(const dims_t& input_shape,
               const dims_t& output_shape);

    void run(runtime_tensor& input_tensor, runtime_tensor& output_tensor);

private:
    std::unique_ptr<ai2d_builder> ai2d_builder_;
    ai2d_datatype_t ai2d_data_type_{};
    ai2d_crop_param_t ai2d_crop_param_{false, 0, 0, 0, 0};
    ai2d_resize_param_t ai2d_resize_param_{false, ai2d_interp_method::tf_bilinear, ai2d_interp_mode::half_pixel};
    ai2d_pad_param_t ai2d_pad_param_{false, {{0, 0}, {0, 0}, {0, 0}, {0, 0}}, ai2d_pad_mode::constant, {0,0,0}};
    ai2d_affine_param_t ai2d_affine_param_{false, ai2d_interp_method::cv2_bilinear, 0, 0, 127, 1, {0.5, 0.1, 0.0, 0.1, 0.5, 0.0}};
    ai2d_shift_param_t ai2d_shift_param_{false, 0};
    dims_t input_shape_;
    dims_t output_shape_;
};
```

---

## AI2D 接口说明

### AI2D 构造函数

```cpp
explicit AI2D();
```

**功能：**
创建 AI2D 对象实例，初始化内部成员。

---

### AI2D 析构函数

```cpp
~AI2D();
```

**功能：**
销毁 AI2D 对象，释放内部资源。

---

### 设置输入输出数据类型

```cpp
void set_ai2d_dtype(ai2d_format input_format,
                    ai2d_format output_format,
                    typecode_t input_dtype,
                    typecode_t output_dtype);
```

**功能：**
设置 AI2D 输入输出数据类型和格式。

**参数说明：**

| 参数名            | 类型            | 描述        |
| -------------- | ------------- | --------- |
| input\_format  | `ai2d_format` | 输入张量的数据格式 |
| output\_format | `ai2d_format` | 输出张量的数据格式 |
| input\_dtype   | `typecode_t`  | 输入张量数据类型  |
| output\_dtype  | `typecode_t`  | 输出张量数据类型  |

**异常：**

* 设置失败：`std::runtime_error`

---

### 设置裁剪参数

```cpp
void set_crop(size_t x, size_t y, size_t width, size_t height);
```

**功能：**
设置图像裁剪区域。

**参数说明：**

| 参数名    | 类型       | 描述        |
| ------ | -------- | --------- |
| x      | `size_t` | 裁剪起点 X 坐标 |
| y      | `size_t` | 裁剪起点 Y 坐标 |
| width  | `size_t` | 裁剪宽度      |
| height | `size_t` | 裁剪高度      |

**异常：**

* 设置失败：`std::runtime_error`

---

### 设置缩放参数

```cpp
void set_resize(ai2d_interp_method interp_method,
                ai2d_interp_mode interp_mode);
```

**功能：**
设置缩放插值方法和模式。

**参数说明：**

| 参数名            | 类型                   | 描述   |
| -------------- | -------------------- | ---- |
| interp\_method | `ai2d_interp_method` | 插值方法 |
| interp\_mode   | `ai2d_interp_mode`   | 插值模式 |

**异常：**

* 设置失败：`std::runtime_error`

---

### 设置填充参数

```cpp
void set_pad(const vector<int>& pad,
             ai2d_pad_mode pad_mode,
             const vector<int>& pad_value);
```

**功能：**
设置图像填充参数。

**参数说明：**

| 参数名        | 类型              | 描述                 |
| ---------- | --------------- | ------------------ |
| pad        | `vector<int>`   | 每个维度的填充量（至少 8 个元素） |
| pad\_mode  | `ai2d_pad_mode` | 填充模式               |
| pad\_value | `vector<int>`   | 填充值                |

**异常：**

* pad 元素不足：`std::invalid_argument`
* 设置失败：`std::runtime_error`

---

### 设置仿射变换参数

```cpp
void set_affine(ai2d_interp_method interp_method,
                int cord_round,
                int bound_ind,
                int bound_val,
                int bound_smooth,
                const vector<float>& affine_matrix);
```

**功能：**
设置仿射变换参数，用于旋转、缩放、平移等操作。

**参数说明：**

| 参数名            | 类型                   | 描述          |
| -------------- | -------------------- | ----------- |
| interp\_method | `ai2d_interp_method` | 插值方法        |
| cord\_round    | `int`                | 坐标四舍五入方式    |
| bound\_ind     | `int`                | 边界索引类型      |
| bound\_val     | `int`                | 边界值         |
| bound\_smooth  | `int`                | 边界平滑参数      |
| affine\_matrix | `vector<float>`      | 仿射矩阵（6 个元素） |

**异常：**

* affine\_matrix 元素不足：`std::invalid_argument`
* 设置失败：`std::runtime_error`

---

### 设置移位参数

```cpp
void set_shift(int shift_val);
```

**功能：**
设置图像像素数据的移位值。

**参数说明：**

| 参数名        | 类型    | 描述  |
| ---------- | ----- | --- |
| shift\_val | `int` | 移位值 |

**异常：**

* 设置失败：`std::runtime_error`

---

### 构建 AI2D 流程

```cpp
void build(const dims_t& input_shape,
           const dims_t& output_shape);
```

**功能：**
根据输入输出张量形状构建 AI2D 执行流程。

**参数说明：**

| 参数名           | 类型       | 描述     |
| ------------- | -------- | ------ |
| input\_shape  | `dims_t` | 输入张量形状 |
| output\_shape | `dims_t` | 输出张量形状 |

**异常：**

* 构建失败：`std::runtime_error`

---

### 执行 AI2D 流程

```cpp
void run(runtime_tensor& input_tensor, runtime_tensor& output_tensor);
```

**功能：**
执行 AI2D 预处理流程，将输入张量转换为指定输出张量。

**参数说明：**

| 参数名            | 类型               | 描述     |
| -------------- | ---------------- | ------ |
| input\_tensor  | `runtime_tensor` | 输入张量对象 |
| output\_tensor | `runtime_tensor` | 输出张量对象 |

**异常：**

* 构建未初始化或执行失败：`std::runtime_error`

---
