## Arduino 时间函数 API 手册

### 1. 概述

时间函数提供了程序执行中的延时和计时功能。这些函数是 Arduino 编程的基础，用于控制程序执行节奏、测量时间间隔等。K230 平台提供了毫秒和微秒级的时间控制精度。

### 2. API 介绍

#### 2.1 `delay` 函数

```cpp
void delay(unsigned long ms)
```

暂停程序执行指定的毫秒数。

**参数**

- `ms`: 延时时间（毫秒）

**返回值**
无

#### 2.2 `delayMicroseconds` 函数

```cpp
void delayMicroseconds(unsigned int us)
```

暂停程序执行指定的微秒数。

**参数**

- `us`: 延时时间（微秒）

**返回值**
无

#### 2.3 `millis` 函数

```cpp
unsigned long millis(void)
```

返回程序开始运行以来的毫秒数。

**参数**
无

**返回值**
自程序启动以来的毫秒数

#### 2.4 `micros` 函数

```cpp
unsigned long micros(void)
```

返回程序开始运行以来的微秒数。

**参数**
无

**返回值**
自程序启动以来的微秒数

### 3. 使用说明

1. **延时函数特性**：
   - `delay()` 会阻塞程序执行
   - `delayMicroseconds()` 在短延时时使用忙等待
   - 两个函数都会处理系统信号中断

2. **计时函数特性**：
   - `millis()` 和 `micros()` 不会阻塞
   - 适合测量时间间隔

3. **精度说明**：
   - 毫秒级函数精度约为 1ms
   - 微秒级函数精度受系统调度影响
   - 实际精度可能因系统负载而变化

4. **最佳实践**：
   - 避免在中断服务程序中使用延时函数
   - 使用 `millis()` 实现非阻塞延时

### 4. 示例程序

```cpp
#include "Arduino.h"

const int ledPin = 52;
unsigned long previousMillis = 0;
const long interval = 1000;  // 1秒间隔

void setup() {
    Serial.begin(115200);
    pinMode(ledPin, OUTPUT);
    
    Serial.println("Time Functions Test Started");
    
    // 测试1: 基本延时
    Serial.println("\n1. Testing delay functions:");
    unsigned long start = millis();
    delay(1000);
    unsigned long elapsed = millis() - start;
    Serial.print("delay(1000) took: ");
    Serial.print(elapsed);
    Serial.println(" ms");
    
    // 测试2: 微秒延时
    start = micros();
    delayMicroseconds(1000);
    elapsed = micros() - start;
    Serial.print("delayMicroseconds(1000) took: ");
    Serial.print(elapsed);
    Serial.println(" us");
    
    // 测试3: 测量代码执行时间
    Serial.println("\n2. Measuring code execution time:");
    start = micros();
    // 执行一些操作
    for (int i = 0; i < 1000; i++) {
        digitalRead(ledPin);
    }
    elapsed = micros() - start;
    Serial.print("1000 digitalRead operations took: ");
    Serial.print(elapsed);
    Serial.println(" us");
}

void loop() {
    // 非阻塞LED闪烁示例
    unsigned long currentMillis = millis();
    
    if (currentMillis - previousMillis >= interval) {
        previousMillis = currentMillis;
        
        // 切换LED状态
        static bool ledState = false;
        ledState = !ledState;
        digitalWrite(ledPin, ledState);
        
        // 打印时间戳
        Serial.print("Time: ");
        Serial.print(currentMillis);
        Serial.print(" ms, LED: ");
        Serial.println(ledState ? "ON" : "OFF");
    }
}
```
