## Arduino 数字 I/O API 手册

### 1. 概述

数字 I/O 是 Arduino 最基本的功能之一，用于控制和读取数字引脚的状态。K230 支持最多 64 个 GPIO 引脚，每个引脚都可以配置为输入或输出模式，并支持上拉、下拉电阻以及开漏输出等模式。

### 2. API 介绍

#### 2.1 `pinMode` 函数

```cpp
void pinMode(pin_size_t pinNumber, PinMode pinMode)
```

配置指定引脚的工作模式。

**参数**

- `pinNumber`: 引脚编号（0-63）
- `pinMode`: 引脚模式，可选值：
  - `OUTPUT`: 输出模式
  - `INPUT`: 输入模式
  - `INPUT_PULLUP`: 带上拉电阻的输入模式
  - `INPUT_PULLDOWN`: 带下拉电阻的输入模式
  - `OUTPUT_OPENDRAIN`: 开漏输出模式

**返回值**
无

#### 2.2 `digitalWrite` 函数

```cpp
void digitalWrite(pin_size_t pinNumber, PinStatus status)
```

设置数字引脚的输出电平。

**参数**

- `pinNumber`: 引脚编号（0-63）
- `status`: 输出电平
  - `HIGH`: 高电平
  - `LOW`: 低电平

**返回值**
无

**注意**
调用此函数前应先使用 `pinMode()` 将引脚设置为输出模式。

#### 2.3 `digitalRead` 函数

```cpp
PinStatus digitalRead(pin_size_t pinNumber)
```

读取数字引脚的输入电平。

**参数**

- `pinNumber`: 引脚编号（0-63）

**返回值**

- `HIGH`: 高电平
- `LOW`: 低电平

**注意**
调用此函数前应先使用 `pinMode()` 将引脚设置为输入模式。

#### 2.4 `attachInterrupt` 方法

```cpp
void attachInterrupt(pin_size_t pinNumber, voidFuncPtr callback, PinStatus mode)
```

在特定引脚上绑定中断

**参数**

- `pinNumber`: 引脚编号（0-63）
- `callback`: 中断回调函数
- `mode`: 触发方式，可选值:
  - `LOW`: 低电平触发
  - `HIGH`: 高电平触发
  - `CHANGE`: 双边沿触发
  - `FALLING`: 下降沿触发
  - `RISING`: 上升沿触发

**返回值**
无

#### 2.5 `attachInterruptArg` 方法

```cpp
void attachInterruptArg(pin_size_t pinNumber, voidFuncPtrParam callback, void *param, PinStatus mode)
```

在特定引脚上绑定中断（带参数版本）

**参数**

- `pinNumber`: 引脚编号（0-63）
- `callback`: 中断回调函数
- `param`: 传递给回调函数的参数
- `mode`: 触发方式，可选值:
  - `LOW`: 低电平触发
  - `HIGH`: 高电平触发
  - `CHANGE`: 双边沿触发
  - `FALLING`: 下降沿触发
  - `RISING`: 上升沿触发

**返回值**
无

#### 2.6 `detachInterrupt` 方法

```cpp
void detachInterrupt(pin_size_t pinNumber)
```

在特定引脚上解除中断

**参数**

- `pinNumber`: 引脚编号（0-63）

**返回值**
无

### 3. 使用说明

1. **引脚编号范围**：K230 支持 0-63 共 64 个 GPIO 引脚。

2. **模式配置**：在使用引脚前必须先调用 `pinMode()` 配置引脚模式。

### 4. 示例程序

```cpp
#include "Arduino.h"

const int buttonPin = 21;     // 按钮连接到引脚21
const int ledPin = 52;       // LED连接到引脚52

int buttonState = 0;         // 存储按钮状态

void setup() {
    // 初始化LED引脚为输出模式
    pinMode(ledPin, OUTPUT);
    
    // 初始化按钮引脚为带上拉电阻的输入模式
    pinMode(buttonPin, INPUT_PULLUP);
    
    Serial.begin(115200);
    Serial.println("Digital I/O Test Started");
}

void loop() {
    // 读取按钮状态（注意：使用上拉电阻时，按下为LOW）
    buttonState = digitalRead(buttonPin);
    
    // 根据按钮状态控制LED
    if (buttonState == LOW) {
        // 按钮被按下，点亮LED
        digitalWrite(ledPin, HIGH);
        Serial.println("Button pressed - LED ON");
    } else {
        // 按钮未按下，熄灭LED
        digitalWrite(ledPin, LOW);
        Serial.println("Button released - LED OFF");
    }
    
    delay(100);  // 简单的消抖延时
}
```
