好的，这是基于您提供的 `k230_display.h` 和 `k230_display.cpp` 源文件，遵循您指定格式生成的 `Display` 类接口开发文档。

-----

# Display (视频输出) 接口开发文档

## Display 模块概述

`Display` 类是对 K230 平台 **VO (Video Output)** 模块的 C++ 封装。它提供了一个单例 (Singleton) 模式的接口，用于初始化显示设备、配置视频输出图层（Layer）以及管理帧数据显示。

该类支持多层显示，分为 Video 层（用于硬件绑定）和 OSD 层（用于软件帧插入）。

**主要特性：**

  * **单例模式**：通过 `instance()` 方法获取全局唯一实例.
  * **多层管理**：支持 **Video Layer** (LAYER\_VIDEO1, LAYER\_VIDEO2) 和 **OSD Layer** (LAYER\_OSD0-LAYER\_OSD3) 的配置和控制.
  * **媒体绑定**：支持将其他媒体源（如 VICAP, VENC）通过系统绑定到 Video Layer.
  * **硬件加速**：使用 GDMA 模块为 OSD 层提供的帧数据执行**硬件旋转和镜像**操作.

-----

## Display 类定义

```cpp
class Display {
public:
    // VO Channels (Layer)
    enum Layer {
        LAYER_VIDEO1 = K_VO_DISPLAY_CHN_ID1,
        LAYER_VIDEO2 = K_VO_DISPLAY_CHN_ID2,
        LAYER_OSD0   = K_VO_DISPLAY_CHN_ID3,
        LAYER_OSD1   = K_VO_DISPLAY_CHN_ID4,
        LAYER_OSD2   = K_VO_DISPLAY_CHN_ID5,
        LAYER_OSD3   = K_VO_DISPLAY_CHN_ID6
    };

    // Flags (Rotation, Mirror, Gray)
    enum Flags {
        FLAG_ROTATION_0   = K_ROTATION_0,
        FLAG_ROTATION_90  = K_ROTATION_90,
        FLAG_ROTATION_180 = K_ROTATION_180,
        FLAG_ROTATION_270 = K_ROTATION_270,
        FLAG_MIRROR_NONE  = K_VO_MIRROR_NONE,
        FLAG_MIRROR_HOR   = K_VO_MIRROR_HOR,
        FLAG_MIRROR_VER   = K_VO_MIRROR_VER,
        FLAG_MIRROR_BOTH  = K_VO_MIRROR_BOTH,
        FLAG_GRAY_ENABLE  = K_VO_GRAY_ENABLE,
        FLAG_GRAY_DISABLE = K_VO_GRAY_DISABLE
    };

    static Display& instance();

    Display(const Display&)            = delete;
    Display& operator=(const Display&) = delete;
    Display(Display&&)                 = delete;
    Display& operator=(Display&&)      = delete;

    int init(k_connector_type type, int osd_num = 1, int width = 640, int height = 480, int fps = 90);
    int deinit();

    int bind_layer(std::tuple<int, int, int> src, int layer, std::tuple<int, int, int, int> rect, int pixelFormat, int flag,
                   int alpha);
    int unbind_layer(int layer);
    void disable_layer(int layer);

    int show(k_video_frame_info& src_frame, std::tuple<int, int> pos, int layer, int alpha, int flag);

    int width() const;
    int height() const;
    int fps();
};
```

-----

## Display 类接口说明

### 1\. 核心生命周期与查询

#### 获取单例实例

```cpp
static Display& instance();
```

**功能：**
获取 `Display` 类的全局唯一实例.

**返回值：**

  * `Display&`：Display 对象的引用。

-----

#### 初始化显示系统

```cpp
int init(k_connector_type type, int osd_num = 1, int width = 640, int height = 480, int fps = 90);
```

**功能：**
初始化显示连接器 (Connector) 和 VO 系统. 此函数会配置显示分辨率，创建 OSD 缓冲区所需的 VB Pool，并初始化 GDMA 模块.

**参数：**

| 参数名 | 类型 | 描述 |
| :--- | :--- | :--- |
| `type` | `k_connector_type` | 要初始化的显示连接器类型。 |
| `osd_num` | `int` | 期望的 OSD 层 VB 缓冲区数量。默认值 `1`. |
| `width` | `int` | 期望的显示宽度（主要用于虚拟设备）。默认值 `640`. |
| `height` | `int` | 期望的显示高度（主要用于虚拟设备）。默认值 `480`. |
| `fps` | `int` | 期望的帧率（主要用于虚拟设备）。默认值 `90`. |

**返回值：**

  * `0`：成功。
  * `-1`：失败（如连接器信息获取失败、VB/DMA 初始化失败）.

-----

#### 反初始化显示系统

```cpp
int deinit();
```

**功能：**
反初始化显示系统，禁用所有图层，关闭连接器电源，释放 DMA 资源，并销毁所有 VB Pool.

**返回值：**

  * `0`：成功。

-----

#### 获取显示宽度

```cpp
int width() const;
```

**功能：**
返回当前配置的显示器宽度（像素）.

**返回值：**

  * `int`：显示宽度。

-----

#### 获取显示高度

```cpp
int height() const;
```

**功能：**
返回当前配置的显示器高度（像素）.

**返回值：**

  * `int`：显示高度。

-----

#### 获取显示帧率

```cpp
int fps();
```

**功能：**
获取显示连接器配置的帧率.

**返回值：**

  * `int`：帧率（FPS）。

-----

### 2\. 图层 (Layer) 管理 API

#### 绑定媒体源到图层

```cpp
int bind_layer(std::tuple<int, int, int> src, int layer, std::tuple<int, int, int, int> rect, int pixelFormat, int flag, int alpha);
```

**功能：**
将一个媒体处理单元（如 VICAP, VENC）的输出通道通过 **系统绑定 (sys\_bind)** 连接到指定的 VO 图层.

**参数：**

| 参数名 | 类型 | 描述 |
| :--- | :--- | :--- |
| `src` | `std::tuple<int, int, int>` | **源**模块通道信息：`{k_mod_id, DevID, ChnID}`. |
| `layer` | `int` | 目标 VO 图层 ID（使用 `Display::Layer` 枚举）. |
| `rect` | `std::tuple<int, int, int, int>` | 图层在屏幕上的显示区域：`{X, Y, Width, Height}`. |
| `pixelFormat` | `int` | 像素格式（`k_pixel_format`）。Video Layer **仅支持** `PIXEL_FORMAT_YUV_SEMIPLANAR_420`. |
| `flag` | `int` | 旋转/镜像标志（使用 `Display::Flags` 枚举）. |
| `alpha` | `int` | 图层全局 Alpha 值（0-255）. |

**返回值：**

  * `0`：成功。
  * `-1`：失败（如图层超出范围、Video Layer 格式错误、图层配置失败）.

-----

#### 解除图层绑定

```cpp
int unbind_layer(int layer);
```

**功能：**
解除指定 VO 图层与媒体源之间的系统绑定 (sys\_unbind).

**参数：**

| 参数名 | 类型 | 描述 |
| :--- | :--- | :--- |
| `layer` | `int` | 目标 VO 图层 ID. |

**返回值：**

  * `0`：成功。
  * `-1`：失败（如图层未绑定或超出范围）.

-----

#### 禁用图层

```cpp
void disable_layer(int layer);
```

**功能：**
禁用指定的 Video 或 OSD 图层，停止显示其内容.

-----

### 3\. OSD 帧显示 API

#### 显示软件帧到 OSD

```cpp
int show(k_video_frame_info& src_frame, std::tuple<int, int> pos, int layer, int alpha, int flag);
```

**功能：**
将用户提供的帧数据 `src_frame` 拷贝到内部 VB 缓冲区，并插入到指定的 **OSD 图层**中进行显示。如果 `flag` 指定了旋转或镜像，将通过 GDMA 硬件加速完成操作.

**注意：** 此函数**仅支持 OSD 图层** (`LAYER_OSD0` 到 `LAYER_OSD3`).

**参数：**

| 参数名 | 类型 | 描述 |
| :--- | :--- | :--- |
| `src_frame` | `k_video_frame_info&` | 待显示的帧信息（包含帧数据地址、宽高、格式等）. |
| `pos` | `std::tuple<int, int>` | 帧在屏幕上的左上角坐标：`{X, Y}`. |
| `layer` | `int` | 目标 OSD 图层 ID. |
| `alpha` | `int` | 图层全局 Alpha 值（0-255）. |
| `flag` | `int` | 旋转/镜像标志（使用 `Display::Flags` 枚举）。非 `FLAG_ROTATION_0` 将触发 DMA 硬件操作. |

**返回值：**

  * `0`：成功。
  * `-1`：失败（如图层超出范围、图像格式不支持、DMA 操作失败、或获取内部缓冲区失败）.
